/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.service.http.commons;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.FileDocument;
import eu.europa.esig.dss.service.http.commons.ResourceLoader;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.client.http.DSSFileLoader;
import eu.europa.esig.dss.spi.client.http.DataLoader;
import eu.europa.esig.dss.spi.client.http.Protocol;
import eu.europa.esig.dss.spi.exception.DSSDataLoaderMultipleException;
import eu.europa.esig.dss.spi.exception.DSSExternalResourceException;
import eu.europa.esig.dss.utils.Utils;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCacheDataLoader
implements DataLoader,
DSSFileLoader {
    private static final long serialVersionUID = 1028849693098211169L;
    private static final Logger LOG = LoggerFactory.getLogger(FileCacheDataLoader.class);
    private static final String DATA_LOADER_NOT_CONFIGURED = "The DataLoader is not configured";
    private File fileCacheDirectory = new File(System.getProperty("java.io.tmpdir"));
    private transient ResourceLoader resourceLoader = new ResourceLoader();
    private List<String> toBeLoaded;
    private List<String> toIgnored;
    private long cacheExpirationTime = -1L;
    private DataLoader dataLoader;

    public FileCacheDataLoader() {
    }

    public FileCacheDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public DataLoader getDataLoader() {
        return this.dataLoader;
    }

    public void setDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public void setFileCacheDirectory(File fileCacheDirectory) {
        Objects.requireNonNull(fileCacheDirectory, "File cache directory cannot be null!");
        this.fileCacheDirectory = fileCacheDirectory;
        if (!this.fileCacheDirectory.exists()) {
            if (this.fileCacheDirectory.mkdirs()) {
                LOG.info("A new directory '{}' has been successfully created.", (Object)fileCacheDirectory.getPath());
            } else {
                throw new IllegalStateException(String.format("Unable to create the directory '%s'!", fileCacheDirectory.getPath()));
            }
        }
    }

    public void setCacheExpirationTime(long cacheExpirationTimeInMilliseconds) {
        this.cacheExpirationTime = cacheExpirationTimeInMilliseconds;
    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    public void addToBeLoaded(String url) {
        if (this.toBeLoaded == null) {
            this.toBeLoaded = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank(url)) {
            this.toBeLoaded.add(url);
        }
    }

    public void addToBeIgnored(String urlString) {
        if (this.toIgnored == null) {
            this.toIgnored = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank(urlString)) {
            String normalizedFileName = DSSUtils.getNormalizedString(urlString);
            this.toIgnored.add(normalizedFileName);
        }
    }

    @Override
    public byte[] get(String url, boolean refresh) {
        DSSDocument document = this.getDocument(url, refresh);
        return DSSUtils.toByteArray(document);
    }

    @Override
    public byte[] get(String url) throws DSSException {
        return this.get(url, false);
    }

    private DSSDocument getDocument(String url, boolean refresh) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        if (this.toBeLoaded != null && !this.toBeLoaded.contains(url)) {
            throw new DSSExternalResourceException(String.format("The toBeLoaded list does not contain URL [%s]!", url));
        }
        String fileName = DSSUtils.getNormalizedString(url);
        File file = this.getCacheFile(fileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !refresh && !isCacheExpired) {
            LOG.debug("Cached file was used");
            return new FileDocument(file);
        }
        if (!fileExists) {
            LOG.debug("There is no cached file!");
        } else {
            LOG.debug("The refresh is forced!");
        }
        byte[] bytes = !this.isNetworkProtocol(url) ? this.getLocalFileContent(url) : this.dataLoader.get(url);
        if (Utils.isArrayNotEmpty(bytes)) {
            File out = this.createFile(fileName, bytes);
            return new FileDocument(out);
        }
        throw new DSSExternalResourceException(String.format("Cannot retrieve data from url [%s]. Empty content is obtained!", url));
    }

    @Override
    public DSSDocument getDocument(String url) {
        return this.getDocument(url, false);
    }

    @Override
    public boolean remove(String url) {
        String fileName = DSSUtils.getNormalizedString(url);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Deleting the file corresponding to URL '{}'...", (Object)url);
            }
            return file.delete();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Unable to remove the file corresponding to URL '{}'! The file does not exist.", (Object)url);
        }
        return false;
    }

    protected boolean isNetworkProtocol(String urlString) {
        String normalizedUrl = Utils.trim(urlString).toLowerCase();
        return Protocol.isHttpUrl(normalizedUrl) || Protocol.isLdapUrl(normalizedUrl) || Protocol.isFtpUrl(normalizedUrl);
    }

    private byte[] getLocalFileContent(String urlString) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        String resourcePath = this.resourceLoader.getAbsoluteResourceFolder(Utils.trim(urlString));
        if (resourcePath != null) {
            File fileResource = new File(resourcePath);
            return DSSUtils.toByteArray(fileResource);
        }
        return this.dataLoader.get(urlString);
    }

    private File getCacheFile(String fileName) {
        String trimmedFileName = Utils.trim(fileName);
        if (this.toIgnored != null && this.toIgnored.contains(trimmedFileName)) {
            throw new DSSExternalResourceException("Part of urls to ignore.");
        }
        LOG.debug("Cached file: {}/{}", (Object)this.fileCacheDirectory, (Object)trimmedFileName);
        return new File(this.fileCacheDirectory, trimmedFileName);
    }

    public File createFile(String urlString, byte[] bytes) {
        String fileName = DSSUtils.getNormalizedString(urlString);
        File file = this.getCacheFile(fileName);
        DSSUtils.saveToFile(bytes, file);
        return file;
    }

    public byte[] loadFileFromCache(String urlString) throws DSSException {
        String fileName = DSSUtils.getNormalizedString(urlString);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            return DSSUtils.toByteArray(file);
        }
        throw new DSSExternalResourceException(String.format("The file with URL [%s] does not exist in the cache!", urlString));
    }

    @Override
    public byte[] post(String urlString, byte[] content) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        String fileName = DSSUtils.getNormalizedString(urlString);
        byte[] digest = DSSUtils.digest(DigestAlgorithm.MD5, content);
        String digestHexEncoded = DSSUtils.toHex(digest);
        String cacheFileName = fileName + "." + digestHexEncoded;
        File file = this.getCacheFile(cacheFileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !isCacheExpired) {
            LOG.debug("Cached file was used");
            return DSSUtils.toByteArray(file);
        }
        LOG.debug("There is no cached file!");
        byte[] returnedBytes = null;
        if (this.isNetworkProtocol(urlString)) {
            returnedBytes = this.dataLoader.post(urlString, content);
        }
        if (Utils.isArrayNotEmpty(returnedBytes)) {
            File cacheFile = this.getCacheFile(cacheFileName);
            DSSUtils.saveToFile(returnedBytes, cacheFile);
            return returnedBytes;
        }
        throw new DSSExternalResourceException(String.format("Cannot retrieve data from URL [%s]", urlString));
    }

    private boolean isCacheExpired(File file) {
        if (this.cacheExpirationTime < 0L) {
            return false;
        }
        if (!file.exists()) {
            return true;
        }
        long currentTime = new Date().getTime();
        if (currentTime - file.lastModified() >= this.cacheExpirationTime) {
            LOG.debug("Cache is expired");
            return true;
        }
        return false;
    }

    @Override
    public DataLoader.DataAndUrl get(List<String> urlStrings) {
        if (Utils.isCollectionEmpty(urlStrings)) {
            throw new DSSExternalResourceException("Cannot process the GET call. List of URLs is empty!");
        }
        HashMap<String, Throwable> exceptions = new HashMap<String, Throwable>();
        for (String urlString : urlStrings) {
            LOG.debug("Processing a GET call to URL [{}]...", (Object)urlString);
            try {
                byte[] bytes = this.get(urlString);
                if (Utils.isArrayEmpty(bytes)) {
                    LOG.debug("The retrieved content from URL [{}] is empty. Continue with other URLs...", (Object)urlString);
                    continue;
                }
                return new DataLoader.DataAndUrl(urlString, bytes);
            }
            catch (Exception e) {
                LOG.warn("Cannot obtain data using '{}' : {}", (Object)urlString, (Object)e.getMessage());
                exceptions.put(urlString, e);
            }
        }
        throw new DSSDataLoaderMultipleException(exceptions);
    }

    @Override
    public void setContentType(String contentType) {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        this.dataLoader.setContentType(contentType);
    }
}

