/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCContent;
import eu.europa.esig.dss.asic.common.ASiCParameters;
import eu.europa.esig.dss.asic.common.ContainerEntryDocument;
import eu.europa.esig.dss.asic.common.ZipUtils;
import eu.europa.esig.dss.enumerations.ASiCContainerType;
import eu.europa.esig.dss.enumerations.MimeType;
import eu.europa.esig.dss.enumerations.MimeTypeEnum;
import eu.europa.esig.dss.exception.IllegalInputException;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.validation.ManifestEntry;
import eu.europa.esig.dss.validation.ManifestFile;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ASiCUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ASiCUtils.class);
    public static final String MANIFEST_FILENAME = "Manifest";
    public static final String ASIC_MANIFEST_FILENAME = "ASiCManifest";
    public static final String ASIC_ARCHIVE_MANIFEST_FILENAME = "ASiCArchiveManifest";
    public static final String ASIC_XAdES_MANIFEST_FILENAME = "manifest";
    public static final String MIME_TYPE = "mimetype";
    public static final String MIME_TYPE_COMMENT = "mimetype=";
    public static final String META_INF_FOLDER = "META-INF/";
    public static final String PACKAGE_ZIP = "package.zip";
    public static final String SIGNATURE_FILENAME = "signature";
    public static final String SIGNATURES_FILENAME = "signatures";
    public static final String TIMESTAMP_FILENAME = "timestamp";
    public static final String CADES_SIGNATURE_EXTENSION = ".p7s";
    public static final String TST_EXTENSION = ".tst";
    public static final String XML_EXTENSION = ".xml";
    public static final String SIGNATURES_XML = "META-INF/signatures.xml";
    public static final String OPEN_DOCUMENT_SIGNATURES = "META-INF/documentsignatures.xml";
    public static final String ASICE_METAINF_MANIFEST = "META-INF/manifest.xml";
    public static final String ASICE_METAINF_XADES_SIGNATURE = "META-INF/signatures001.xml";
    public static final String ASICE_METAINF_CADES_SIGNATURE = "META-INF/signature001.p7s";
    public static final String ASICE_METAINF_CADES_TIMESTAMP = "META-INF/timestamp001.tst";
    public static final String ASICE_METAINF_CADES_MANIFEST = "META-INF/ASiCManifest001.xml";
    public static final String ASICE_METAINF_CADES_ARCHIVE_MANIFEST = "META-INF/ASiCArchiveManifest001.xml";
    public static final String SIGNATURE_P7S = "META-INF/signature.p7s";
    public static final String TIMESTAMP_TST = "META-INF/timestamp.tst";

    private ASiCUtils() {
    }

    public static boolean isSignature(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(SIGNATURE_FILENAME) && !entryName.contains(MANIFEST_FILENAME);
    }

    public static boolean isTimestamp(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(TIMESTAMP_FILENAME) && entryName.endsWith(TST_EXTENSION);
    }

    public static String getMimeTypeString(ASiCParameters asicParameters) {
        MimeType mimeType = ASiCUtils.getMimeType(asicParameters);
        return mimeType.getMimeTypeString();
    }

    public static String getZipComment(ASiCParameters asicParameters) {
        if (asicParameters.isZipComment()) {
            return ASiCUtils.getZipComment(ASiCUtils.getMimeTypeString(asicParameters));
        }
        return "";
    }

    public static String getZipComment(MimeType mimeType) {
        return ASiCUtils.getZipComment(mimeType.getMimeTypeString());
    }

    public static String getZipComment(String mimeTypeString) {
        return MIME_TYPE_COMMENT + mimeTypeString;
    }

    public static boolean isASiCMimeType(MimeType mimeType) {
        return MimeTypeEnum.ASICS.equals(mimeType) || MimeTypeEnum.ASICE.equals(mimeType);
    }

    public static boolean isOpenDocumentMimeType(MimeType mimeType) {
        return MimeTypeEnum.ODT.equals(mimeType) || MimeTypeEnum.ODS.equals(mimeType) || MimeTypeEnum.ODG.equals(mimeType) || MimeTypeEnum.ODP.equals(mimeType);
    }

    public static ASiCContainerType getASiCContainerType(MimeType asicMimeType) {
        Objects.requireNonNull(asicMimeType, "MimeType cannot be null!");
        if (MimeTypeEnum.ASICS.equals(asicMimeType)) {
            return ASiCContainerType.ASiC_S;
        }
        if (MimeTypeEnum.ASICE.equals(asicMimeType) || ASiCUtils.isOpenDocumentMimeType(asicMimeType)) {
            return ASiCContainerType.ASiC_E;
        }
        throw new IllegalArgumentException("Not allowed mimetype '" + asicMimeType.getMimeTypeString() + "'");
    }

    public static boolean isASiCE(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_E.equals((Object)asicParameters.getContainerType());
    }

    public static boolean isASiCS(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_S.equals((Object)asicParameters.getContainerType());
    }

    public static MimeType getMimeType(ASiCParameters asicParameters) {
        if (Utils.isStringNotBlank(asicParameters.getMimeType())) {
            return MimeType.fromMimeTypeString(asicParameters.getMimeType());
        }
        return ASiCUtils.isASiCE(asicParameters) ? MimeTypeEnum.ASICE : MimeTypeEnum.ASICS;
    }

    public static boolean filesContainCorrectSignatureFileWithExtension(List<String> filenames, String extension) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename) || !filename.endsWith(extension)) continue;
            return true;
        }
        return false;
    }

    public static boolean filesContainSignatures(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isAsicFileContent(List<String> filenames) {
        return ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION) || ASiCUtils.filesContainTimestamps(filenames);
    }

    public static boolean filesContainTimestamps(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isTimestamp(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isZip(DSSDocument document) {
        boolean bl;
        block9: {
            if (document == null) {
                return false;
            }
            InputStream is = document.openStream();
            try {
                bl = ASiCUtils.isZip(is);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new DSSException(String.format("Unable to determine whether the document with name '%s' represents a ZIP container. Reason : %s", document.getName(), e.getMessage()), e);
                }
            }
            is.close();
        }
        return bl;
    }

    public static boolean isZip(InputStream is) {
        Objects.requireNonNull(is, "InputStream cannot be null!");
        byte[] preamble = new byte[2];
        try {
            int r = is.read(preamble, 0, 2);
            if (r != 2) {
                return false;
            }
        }
        catch (IOException e) {
            throw new IllegalInputException("Unable to read the 2 first bytes", e);
        }
        return preamble[0] == 80 && preamble[1] == 75;
    }

    public static boolean isASiCWithXAdES(List<String> filenames) {
        return ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION);
    }

    public static boolean isASiCWithCAdES(List<String> filenames) {
        return ASiCUtils.filesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.filesContainTimestamps(filenames);
    }

    public static boolean isXAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(XML_EXTENSION);
    }

    public static boolean isCAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(CADES_SIGNATURE_EXTENSION);
    }

    public static boolean isContainerOpenDocument(DSSDocument archiveContainer) {
        DSSDocument mimetype = ASiCUtils.getMimetypeDocument(archiveContainer);
        return mimetype != null && ASiCUtils.isOpenDocument(mimetype);
    }

    private static DSSDocument getMimetypeDocument(DSSDocument archiveDocument) {
        List<DSSDocument> documents = ZipUtils.getInstance().extractContainerContent(archiveDocument);
        for (DSSDocument document : documents) {
            if (!ASiCUtils.isMimetype(document.getName())) continue;
            return document;
        }
        return null;
    }

    public static boolean isOpenDocument(DSSDocument mimeTypeDoc) {
        MimeType mimeType = ASiCUtils.getMimeType(mimeTypeDoc);
        if (mimeTypeDoc != null) {
            return ASiCUtils.isOpenDocumentMimeType(mimeType);
        }
        return false;
    }

    public static boolean areFilesContainMimetype(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isMimetype(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isMimetype(String entryName) {
        return MIME_TYPE.equals(entryName);
    }

    public static MimeType getMimeType(DSSDocument mimeTypeDocument) {
        MimeType mimeType;
        block9: {
            if (mimeTypeDocument == null) {
                return null;
            }
            InputStream is = mimeTypeDocument.openStream();
            try {
                byte[] byteArray = Utils.toByteArray(is);
                String mimeTypeString = new String(byteArray, StandardCharsets.UTF_8);
                mimeType = MimeType.fromMimeTypeString(mimeTypeString);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new IllegalInputException(String.format("Unable to read mimetype file. Reason : %s", e.getMessage()), e);
                }
            }
            is.close();
        }
        return mimeType;
    }

    public static boolean isASiCSContainer(DSSDocument container) {
        return ASiCContainerType.ASiC_S.equals((Object)ASiCUtils.getContainerType(container));
    }

    public static boolean isASiCEContainer(DSSDocument container) {
        return ASiCContainerType.ASiC_E.equals((Object)ASiCUtils.getContainerType(container));
    }

    public static ASiCContainerType getContainerType(DSSDocument archiveContainer) {
        List<String> entryNames = ZipUtils.getInstance().extractEntryNames(archiveContainer);
        DSSDocument mimetypeDocument = null;
        if (ASiCUtils.areFilesContainMimetype(entryNames)) {
            mimetypeDocument = ASiCUtils.getMimetypeDocument(archiveContainer);
        }
        String zipComment = ASiCUtils.getZipComment(archiveContainer);
        int signedDocumentsNumber = ASiCUtils.getNumberOfSignedRootDocuments(entryNames);
        return ASiCUtils.getContainerType(archiveContainer.getMimeType(), mimetypeDocument, zipComment, signedDocumentsNumber);
    }

    public static boolean isASiCSContainer(ASiCContent asicContent) {
        return ASiCContainerType.ASiC_S.equals((Object)ASiCUtils.getContainerType(asicContent));
    }

    public static boolean isASiCEContainer(ASiCContent asicContent) {
        return ASiCContainerType.ASiC_E.equals((Object)ASiCUtils.getContainerType(asicContent));
    }

    public static ASiCContainerType getContainerType(ASiCContent asicContent) {
        if (asicContent.getContainerType() != null) {
            return asicContent.getContainerType();
        }
        return ASiCUtils.getContainerType(asicContent.getAsicContainer().getMimeType(), asicContent.getMimeTypeDocument(), asicContent.getZipComment(), Utils.collectionSize(asicContent.getRootLevelSignedDocuments()));
    }

    private static int getNumberOfSignedRootDocuments(List<String> containerEntryNames) {
        int signedDocumentCounter = 0;
        for (String documentName : containerEntryNames) {
            if (documentName.contains("/") || ASiCUtils.isMimetype(documentName)) continue;
            ++signedDocumentCounter;
        }
        return signedDocumentCounter;
    }

    private static ASiCContainerType getContainerType(MimeType containerMimeType, DSSDocument mimetypeDocument, String zipComment, int rootSignedDocumentsNumber) {
        ASiCContainerType containerType = ASiCUtils.getContainerTypeFromMimeType(containerMimeType);
        if (containerType == null && (containerType = ASiCUtils.getContainerTypeFromMimeTypeDocument(mimetypeDocument)) == null) {
            containerType = ASiCUtils.getContainerTypeFromZipComment(zipComment);
        }
        if (containerType == null) {
            LOG.info("Unable to define the ASiC Container type with its properties. Assume type based on root-level documents...");
            if (rootSignedDocumentsNumber == 1) {
                containerType = ASiCContainerType.ASiC_S;
            } else if (rootSignedDocumentsNumber > 1) {
                containerType = ASiCContainerType.ASiC_E;
            } else {
                LOG.warn("The provided container does not contain signer documents on the root level!");
            }
        }
        return containerType;
    }

    private static ASiCContainerType getContainerTypeFromZipComment(String zipComment) {
        int indexOf;
        if (Utils.isStringNotBlank(zipComment) && (indexOf = zipComment.indexOf(MIME_TYPE_COMMENT)) > -1) {
            String asicCommentMimeTypeString = zipComment.substring(MIME_TYPE_COMMENT.length() + indexOf);
            MimeType mimeTypeFromZipComment = MimeType.fromMimeTypeString(asicCommentMimeTypeString);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromZipComment);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeTypeDocument(DSSDocument mimetype) {
        if (mimetype != null) {
            MimeType mimeTypeFromEmbeddedFile = ASiCUtils.getMimeType(mimetype);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromEmbeddedFile);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeType(MimeType mimeType) {
        if (ASiCUtils.isASiCMimeType(mimeType)) {
            return ASiCUtils.getASiCContainerType(mimeType);
        }
        return null;
    }

    public static boolean isManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static boolean isArchiveManifest(String fileName) {
        return fileName.startsWith(META_INF_FOLDER) && fileName.contains(ASIC_ARCHIVE_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static boolean coversSignature(ManifestFile manifestFile) {
        for (ManifestEntry manifestEntry : manifestFile.getEntries()) {
            if (!ASiCUtils.isSignature(manifestEntry.getFileName())) continue;
            return true;
        }
        return false;
    }

    public static void addOrReplaceDocument(List<DSSDocument> documentList, DSSDocument newDocument) {
        for (int i = 0; i < documentList.size(); ++i) {
            if (!newDocument.getName().equals(documentList.get(i).getName())) continue;
            documentList.set(i, newDocument);
            return;
        }
        documentList.add(newDocument);
    }

    public static ASiCContent ensureMimeTypeAndZipComment(ASiCContent asicContent, ASiCParameters asicParameters) {
        if (asicContent.getMimeTypeDocument() == null) {
            MimeType mimeType = ASiCUtils.getMimeType(asicContent, asicParameters);
            DSSDocument mimetypeDocument = ASiCUtils.createMimetypeDocument(mimeType);
            asicContent.setMimeTypeDocument(mimetypeDocument);
        }
        if (Utils.isStringEmpty(asicContent.getZipComment())) {
            String zipComment = ASiCUtils.getZipComment(asicContent, asicParameters);
            asicContent.setZipComment(zipComment);
        }
        return asicContent;
    }

    private static MimeType getMimeType(ASiCContent asicContent, ASiCParameters asicParameters) {
        MimeType mimeType = null;
        DSSDocument mimeTypeDocument = asicContent.getMimeTypeDocument();
        if (mimeTypeDocument != null) {
            mimeType = ASiCUtils.getMimeType(mimeTypeDocument);
        }
        if (mimeType == null) {
            Objects.requireNonNull(asicParameters, "ASiCParameters shall be present for the requested operation!");
            mimeType = ASiCUtils.getMimeType(asicParameters);
        }
        return mimeType;
    }

    private static String getZipComment(ASiCContent asicContent, ASiCParameters asicParameters) {
        String zipComment = asicContent.getZipComment();
        if (Utils.isStringNotEmpty(zipComment)) {
            return zipComment;
        }
        if (asicParameters != null) {
            return ASiCUtils.getZipComment(asicParameters);
        }
        return "";
    }

    private static DSSDocument createMimetypeDocument(MimeType mimeType) {
        byte[] mimeTypeBytes = mimeType.getMimeTypeString().getBytes(StandardCharsets.UTF_8);
        InMemoryDocument mimetypeDocument = new InMemoryDocument(mimeTypeBytes, MIME_TYPE);
        ContainerEntryDocument zipEntryDocument = new ContainerEntryDocument(mimetypeDocument);
        zipEntryDocument.getZipEntry().setCompressionMethod(0);
        return zipEntryDocument;
    }

    public static List<DSSDocument> getRootLevelSignedDocuments(ASiCContent asicContent) {
        List<DSSDocument> signedDocuments = asicContent.getSignedDocuments();
        if (Utils.isCollectionEmpty(signedDocuments)) {
            return Collections.emptyList();
        }
        if (Utils.collectionSize(signedDocuments) == 1) {
            return signedDocuments;
        }
        return ASiCUtils.getRootLevelDocuments(signedDocuments);
    }

    public static List<DSSDocument> getRootLevelDocuments(List<DSSDocument> documents) {
        ArrayList<DSSDocument> rootDocuments = new ArrayList<DSSDocument>();
        for (DSSDocument document : documents) {
            String documentName = document.getName();
            if (documentName == null || documentName.contains("/") || MIME_TYPE.equals(documentName)) continue;
            rootDocuments.add(document);
        }
        return rootDocuments;
    }

    public static String getZipComment(DSSDocument archiveContainer) {
        byte[] buffer = DSSUtils.toByteArray(archiveContainer);
        if (Utils.isArrayEmpty(buffer)) {
            LOG.warn("An empty container obtained! Unable to extract zip comment.");
            return null;
        }
        int len = buffer.length;
        byte[] magicDirEnd = new byte[]{80, 75, 5, 6};
        for (int ii = len - 22; ii >= 0; --ii) {
            boolean isMagicStart = true;
            for (int jj = 0; jj < magicDirEnd.length; ++jj) {
                if (buffer[ii + jj] == magicDirEnd[jj]) continue;
                isMagicStart = false;
                break;
            }
            if (!isMagicStart) continue;
            int commentLen = buffer[ii + 20] + buffer[ii + 21] * 256;
            int realLen = len - ii - 22;
            if (commentLen != realLen) {
                LOG.warn("WARNING! ZIP comment size mismatch: directory says len is {}, but file ends after {} bytes!", (Object)commentLen, (Object)realLen);
            }
            if (realLen == 0) {
                return null;
            }
            return new String(buffer, ii + 22, realLen);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Zip comment is not found in the provided container with name '{}'", (Object)archiveContainer.getName());
        }
        return null;
    }

    public static List<ManifestEntry> toSimpleManifestEntries(List<DSSDocument> documents) {
        ArrayList<ManifestEntry> entries = new ArrayList<ManifestEntry>();
        for (DSSDocument document : documents) {
            ManifestEntry entry = new ManifestEntry();
            entry.setFileName(document.getName());
            entry.setMimeType(document.getMimeType());
            entry.setFound(true);
            entries.add(entry);
        }
        return entries;
    }
}

